/**
 * @file   IB49config.c
 * @author StepTechnica
 * @date   Sep. 05, 2025
 * 
 * @brief Evaluation Board 用ソースファイル「IB49ボード設定」
 * 
 * @copyright (C) 2022- StepTechnica Co.,Ltd.
 * 
 * このソフトウェアは、コピー利用、配布、変更の追加、変更を加えたもの再配布、商用利用、有料販売など、どなたも自由にお使いいただくことができます。
 * このソフトウェアには保証はついていません。
 * このソフトウェアを利用したことで問題が起きた際に、ソフトウェアの製作者は一切の責任を負いません。
 * 
 */

/* -----------------------------------------------------------------------------
 * #include
 */

#include "IB49config.h"

#include "main.h"
#include "stm32f4xx_hal.h"
#include <stdbool.h>

/* -----------------------------------------------------------------------------
 * 変数定義
 */
static char	SW7[8];

static short gCntBlink=0;

/* -----------------------------------------------------------------------------
 * 関数宣言
 */
EN_MKY_BUS ReadMkyBus(void);
EN_MKY_BUS GetMkyBus(void);
EN_MKY_ENDIAN GetMkyEndian(void);
void SetAccessEndian(AccessEndian endian);
AccessEndian GetAccessEndian();

char readSW7Data(void);
char readSW7_1(void);
char readSW7_2(void);
char readSW7_3(void);
char readSW7_4(void);
char readSW7_5(void);
char readSW7_6(void);
char readSW7_7(void);
char readSW7_8(void);

void drvLED8(char sw);
void drvLED9(char sw);
void drvLED10(char sw);
void drvLED11(char sw);
void drvLED12(char sw);
void drvLED13(char sw);
void drvLED14(char sw);
void drvLED15(char sw);

void drvMKYRESET(char sw);

char isMODEon(void);
char isInt0on(void);
char isInt1on(void);
char isPFon(void);
void setSPI2_CS(char sw);

void LedSw_main(void);


	/* ----- Port-A -------------------------
		PA0  : CN60-28 = SW7-1
		PA1  : CN60-30 = SW7-2
		PA2  : CN61-35 = SW7-3
		PA3  : CN61-37 = SW7-4
		PA4  : CN60-32 = SW7-5
		PA5  : CN61-11 = SW7-6
		PA6  : CN61-13 = SW7-7
		PA7  : CN61-15 = SW7-8
	 */

char readSW7Data(void)
{
	unsigned short pa = (char)(SW7_1_GPIO_Port->IDR);

	return(pa);
}
/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-1読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_1(void)
{
	return((HAL_GPIO_ReadPin(SW7_1_GPIO_Port, SW7_1_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-2読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_2(void)
{
	return((HAL_GPIO_ReadPin(SW7_2_GPIO_Port, SW7_2_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-3読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_3(void)
{
	return((HAL_GPIO_ReadPin(SW7_3_GPIO_Port, SW7_3_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-4読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_4(void)
{
	return((HAL_GPIO_ReadPin(SW7_4_GPIO_Port, SW7_4_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-5読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_5(void)
{
	return((HAL_GPIO_ReadPin(SW7_5_GPIO_Port, SW7_5_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-6読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_6(void)
{
	return((HAL_GPIO_ReadPin(SW7_6_GPIO_Port, SW7_6_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-7読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_7(void)
{
	return((HAL_GPIO_ReadPin(SW7_7_GPIO_Port, SW7_7_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SW7-8読出し
 * 
 * @note： 
 * 
 * @param  none
 * @retval false ： SW OFF
 * @retval true  ： SW ON
 * 
 */
char readSW7_8(void)
{
	return((HAL_GPIO_ReadPin(SW7_8_GPIO_Port, SW7_8_Pin) == GPIO_PIN_SET)? false : true);
}

	/* ----- Port-B -------------------------
		PB0  : CN60-34 = LD1       : Nucleo
		PB4  : CN61-27 = LED6      
		PB5  : CN61-29 = LED7      
		PB6  : CN61-17 = LED8      
		PB7  : CN60-21 = LED9(LD2) ( Nucleo)
		PB8  : CN61-3  = LED10     
		PB9  : CN61-5  = LED11     
		PB10 : CN61-25 = LED12     
		PB11 : CN61-18 = LED13     
		PB14 : CN61-28 = LD3       : Nucleo
	 */

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED8 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED8(char sw)
{
	HAL_GPIO_WritePin(LED8_GPIO_Port, LED8_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED9 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED9(char sw)
{
	HAL_GPIO_WritePin(LED9_GPIO_Port, LED9_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED10 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED10(char sw)
{
	HAL_GPIO_WritePin(LED10_GPIO_Port, LED10_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED11 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED11(char sw)
{
	HAL_GPIO_WritePin(LED11_GPIO_Port, LED11_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED12 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED12(char sw)
{
	HAL_GPIO_WritePin(LED12_GPIO_Port, LED12_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED13 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED13(char sw)
{
	HAL_GPIO_WritePin(LED13_GPIO_Port, LED13_Pin , sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED14 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED14(char sw)
{
	HAL_GPIO_WritePin(LED14_GPIO_Port, LED14_Pin, sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED15 の制御
 * 
 * @note： 
 * 
 * @param  char sw ： false = 消灯、true = 点灯
 * @retval none
 * 
 */
void drvLED15(char sw)
{
	HAL_GPIO_WritePin(LED15_GPIO_Port, LED15_Pin, sw );
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief： Reset MKY Device
 *
 * @note：
 *
 * @param  char sw ： false = Reset、true = Normal
 * @retval none
 *
 */
void drvMKYRESET(char sw)
{
	HAL_GPIO_WritePin(MKY_RESET_GPIO_Port, MKY_RESET_Pin, sw );
}


/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：nPF 読出し
 *
 * @note： nPF : PE3
 *
 * @param  none
 * @retval false ： Port=High, no interrupt
 * @retval true  ： Port=Low,  interrupt
 *
 */
char isMODEon(void)
{
	return((HAL_GPIO_ReadPin(MODE_GPIO_Port, MODE_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：nINT0 読出し
 * 
 * @note： nNT0 : PE5
 * 
 * @param  none
 * @retval false ： Port=High, no interrupt
 * @retval true  ： Port=Low,  interrupt
 * 
 */
char isInt0on(void)
{
	return((HAL_GPIO_ReadPin(nINT0_GPIO_Port, nINT0_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：nINT1 読出し
 * 
 * @note： nINT1 : PE6
 * 
 * @param  none
 * @retval false ： Port=High, no interrupt
 * @retval true  ： Port=Low,  interrupt
 * 
 */
char isInt1on(void)
{
	return((HAL_GPIO_ReadPin(nINT1_GPIO_Port, nINT1_Pin) == GPIO_PIN_SET)? false : true);
}

/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：nPF 読出し
 *
 * @note： nPF : PE6
 *
 * @param  none
 * @retval false ： Port=High, no interrupt
 * @retval true  ： Port=Low,  interrupt
 *
 */
char isPFon(void)
{
	return((HAL_GPIO_ReadPin(nPF_GPIO_Port, nPF_Pin) == GPIO_PIN_SET)? false : true);
}



/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：SPI2 CS 端子設定
 * 
 * @note： SPI2_NSS : PB12
 * 
 * @param  char sw ： false = output Low、true = output High
 * @retval none
 * 
 */
void setSPI2_CS(char sw)
{
	HAL_GPIO_WritePin(GPIOB, GPIO_PIN_12, sw );
}


/**
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * @brief：LED SW処理
 * 
 * @note： main()から呼ぼれるメインルーチン
 *         SW7 の値を読んで、対応するLEDを制御する
 * 
 * @param  none
 * @retval none
 * 
 */
void LedSw_main(void)
{
	// Raed SW data
	SW7[0] = readSW7_1();
	SW7[1] = readSW7_2();
	SW7[2] = readSW7_3();
	SW7[3] = readSW7_4();
	SW7[4] = readSW7_5();
	SW7[5] = readSW7_6();
	SW7[6] = readSW7_7();
	SW7[7] = readSW7_8();

	// LED Control
	// drvLED15( SW7[0] );
	drvLED14( SW7[1] );
	drvLED13( SW7[2] );
	drvLED12( SW7[3] );
	drvLED11( SW7[4] );
	drvLED10( SW7[5] );
	drvLED9( SW7[6] );
	drvLED8( SW7[7] );
}

void blinkLED1(void)
{
	gCntBlink++;

	if (gCntBlink == 5) {
		drvLED15(true);
	} else if (gCntBlink == 10) {
		drvLED15(false);
		gCntBlink = 0;
	}
}

